/*******************************************************************************
* File Name: Comp_1_VRef.c  
* Version 1.90
*
* Description:
*  This file provides the source code to the API for the 8-bit Voltage DAC 
*  (VDAC8) User Module.
*
* Note:
*  Any unusual or non-standard behavior should be noted here. Other-
*  wise, this section should remain blank.
*
********************************************************************************
* Copyright 2008-2012, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
*******************************************************************************/

#include "cytypes.h"
#include "Comp_1_VRef.h"

#if (CY_PSOC5A)
#include <CyLib.h>
#endif /* CY_PSOC5A */

uint8 Comp_1_VRef_initVar = 0u;

#if (CY_PSOC5A)
    static uint8 Comp_1_VRef_restoreVal = 0u;
#endif /* CY_PSOC5A */

#if (CY_PSOC5A)
    static Comp_1_VRef_backupStruct Comp_1_VRef_backup;
#endif /* CY_PSOC5A */


/*******************************************************************************
* Function Name: Comp_1_VRef_Init
********************************************************************************
* Summary:
*  Initialize to the schematic state.
* 
* Parameters:
*  void:
*
* Return:
*  void
*
* Theory:
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_Init(void) 
{
    Comp_1_VRef_CR0 = (Comp_1_VRef_MODE_V );

    /* Set default data source */
    #if(Comp_1_VRef_DEFAULT_DATA_SRC != 0 )
        Comp_1_VRef_CR1 = (Comp_1_VRef_DEFAULT_CNTL | Comp_1_VRef_DACBUS_ENABLE) ;
    #else
        Comp_1_VRef_CR1 = (Comp_1_VRef_DEFAULT_CNTL | Comp_1_VRef_DACBUS_DISABLE) ;
    #endif /* (Comp_1_VRef_DEFAULT_DATA_SRC != 0 ) */

    /* Set default strobe mode */
    #if(Comp_1_VRef_DEFAULT_STRB != 0)
        Comp_1_VRef_Strobe |= Comp_1_VRef_STRB_EN ;
    #endif/* (Comp_1_VRef_DEFAULT_STRB != 0) */

    /* Set default range */
    Comp_1_VRef_SetRange(Comp_1_VRef_DEFAULT_RANGE); 

    /* Set default speed */
    Comp_1_VRef_SetSpeed(Comp_1_VRef_DEFAULT_SPEED);
}


/*******************************************************************************
* Function Name: Comp_1_VRef_Enable
********************************************************************************
* Summary:
*  Enable the VDAC8
* 
* Parameters:
*  void
*
* Return:
*  void
*
* Theory:
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_Enable(void) 
{
    Comp_1_VRef_PWRMGR |= Comp_1_VRef_ACT_PWR_EN;
    Comp_1_VRef_STBY_PWRMGR |= Comp_1_VRef_STBY_PWR_EN;

    /*This is to restore the value of register CR0 ,
    which is modified  in Stop API , this prevents misbehaviour of VDAC */
    #if (CY_PSOC5A)
        if(Comp_1_VRef_restoreVal == 1u) 
        {
             Comp_1_VRef_CR0 = Comp_1_VRef_backup.data_value;
             Comp_1_VRef_restoreVal = 0u;
        }
    #endif /* CY_PSOC5A */
}


/*******************************************************************************
* Function Name: Comp_1_VRef_Start
********************************************************************************
*
* Summary:
*  The start function initializes the voltage DAC with the default values, 
*  and sets the power to the given level.  A power level of 0, is the same as
*  executing the stop function.
*
* Parameters:
*  Power: Sets power level between off (0) and (3) high power
*
* Return:
*  void 
*
* Global variables:
*  Comp_1_VRef_initVar: Is modified when this function is called for the 
*  first time. Is used to ensure that initialization happens only once.
*
*******************************************************************************/
void Comp_1_VRef_Start(void)  
{
    /* Hardware initiazation only needs to occure the first time */
    if(Comp_1_VRef_initVar == 0u)
    { 
        Comp_1_VRef_Init();
        Comp_1_VRef_initVar = 1u;
    }

    /* Enable power to DAC */
    Comp_1_VRef_Enable();

    /* Set default value */
    Comp_1_VRef_SetValue(Comp_1_VRef_DEFAULT_DATA); 
}


/*******************************************************************************
* Function Name: Comp_1_VRef_Stop
********************************************************************************
*
* Summary:
*  Powers down DAC to lowest power state.
*
* Parameters:
*  void
*
* Return:
*  void
*
* Theory:
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_Stop(void) 
{
    /* Disble power to DAC */
    Comp_1_VRef_PWRMGR &= (uint8)(~Comp_1_VRef_ACT_PWR_EN);
    Comp_1_VRef_STBY_PWRMGR &= (uint8)(~Comp_1_VRef_STBY_PWR_EN);

    /* This is a work around for PSoC5A  ,
    this sets VDAC to current mode with output off */
    #if (CY_PSOC5A)
        Comp_1_VRef_backup.data_value = Comp_1_VRef_CR0;
        Comp_1_VRef_CR0 = Comp_1_VRef_CUR_MODE_OUT_OFF;
        Comp_1_VRef_restoreVal = 1u;
    #endif /* CY_PSOC5A */
}


/*******************************************************************************
* Function Name: Comp_1_VRef_SetSpeed
********************************************************************************
*
* Summary:
*  Set DAC speed
*
* Parameters:
*  power: Sets speed value
*
* Return:
*  void
*
* Theory:
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_SetSpeed(uint8 speed) 
{
    /* Clear power mask then write in new value */
    Comp_1_VRef_CR0 &= (uint8)(~Comp_1_VRef_HS_MASK);
    Comp_1_VRef_CR0 |=  (speed & Comp_1_VRef_HS_MASK);
}


/*******************************************************************************
* Function Name: Comp_1_VRef_SetRange
********************************************************************************
*
* Summary:
*  Set one of three current ranges.
*
* Parameters:
*  Range: Sets one of Three valid ranges.
*
* Return:
*  void 
*
* Theory:
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_SetRange(uint8 range) 
{
    Comp_1_VRef_CR0 &= (uint8)(~Comp_1_VRef_RANGE_MASK);      /* Clear existing mode */
    Comp_1_VRef_CR0 |= (range & Comp_1_VRef_RANGE_MASK);      /*  Set Range  */
    Comp_1_VRef_DacTrim();
}


/*******************************************************************************
* Function Name: Comp_1_VRef_SetValue
********************************************************************************
*
* Summary:
*  Set 8-bit DAC value
*
* Parameters:  
*  value:  Sets DAC value between 0 and 255.
*
* Return: 
*  void 
*
* Theory: 
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_SetValue(uint8 value) 
{
    #if (CY_PSOC5A)
        uint8 Comp_1_VRef_intrStatus = CyEnterCriticalSection();
    #endif /* CY_PSOC5A */

    Comp_1_VRef_Data = value;                /*  Set Value  */

    /* PSOC5A requires a double write */
    /* Exit Critical Section */
    #if (CY_PSOC5A)
        Comp_1_VRef_Data = value;
        CyExitCriticalSection(Comp_1_VRef_intrStatus);
    #endif /* CY_PSOC5A */
}


/*******************************************************************************
* Function Name: Comp_1_VRef_DacTrim
********************************************************************************
*
* Summary:
*  Set the trim value for the given range.
*
* Parameters:
*  range:  1V or 4V range.  See constants.
*
* Return:
*  void
*
* Theory: 
*
* Side Effects:
*
*******************************************************************************/
void Comp_1_VRef_DacTrim(void) 
{
    uint8 mode;

    mode = (uint8)((Comp_1_VRef_CR0 & Comp_1_VRef_RANGE_MASK) >> 2) + Comp_1_VRef_TRIM_M7_1V_RNG_OFFSET;
    Comp_1_VRef_TR = CY_GET_XTND_REG8((uint8 *)(Comp_1_VRef_DAC_TRIM_BASE + mode));
}


/* [] END OF FILE */
