/*******************************************************************************
* File Name: I2C.h
* Version 1.90
*
* Description:
*  This is the header file for the EzI2C user module.  It contains function
*  prototypes and constants for the users convenience.
*
********************************************************************************
* Copyright 2008-2013, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#if !defined(CY_EZI2C_I2C_H)
#define CY_EZI2C_I2C_H

#include "cyfitter.h"
#include "cytypes.h"
#include "CyLib.h"

/* Check if required defines such as CY_PSOC5LP are available in cy_boot */
#if !defined (CY_PSOC5LP)
    #error Component EZI2C_v1_90 requires cy_boot v3.0 or later
#endif /* (CY_PSOC5LP) */


/***************************************
*   Conditional Compilation Parameters
***************************************/

#define I2C_ADDRESSES         (1u)
#define I2C_ONE_ADDRESS       (0x01u)
#define I2C_TWO_ADDRESSES     (0x02u)
#define I2C_ENABLE_WAKEUP     (1u)

/* Wakeup enabled */
#define I2C_WAKEUP_ENABLED     (0u != I2C_ENABLE_WAKEUP)


/* Enumerated type*/ 
#define I2C__ANY 0
#define I2C__I2C0 1
#define I2C__I2C1 2


/***************************************
*       Type Definitions
***************************************/

/* Structure to save registers before go to sleep */
typedef struct 
{
    uint8   enableState;

    uint8   xcfg;
    uint8   adr;
    uint8   cfg;
       
    uint8   clkDiv1;
    uint8   clkDiv2;
        
}   I2C_BACKUP_STRUCT;


/***************************************
*        Function Prototypes
***************************************/

void    I2C_Start(void) ;
void    I2C_Stop(void) ;
void    I2C_EnableInt(void) ;
void    I2C_DisableInt(void) ;

void    I2C_SetAddress1(uint8 address) ;
uint8   I2C_GetAddress1(void) ;
void    I2C_SetBuffer1(uint16 bufSize, uint16 rwBoundry, volatile uint8 * dataPtr) ;
uint8   I2C_GetActivity(void) ;
void    I2C_Init(void) ;
void    I2C_Enable(void) ;
void    I2C_Sleep(void) ;
void    I2C_Wakeup(void) ;
void    I2C_SaveConfig(void) ;
void    I2C_RestoreConfig(void) ;


#if(I2C_ADDRESSES == I2C_TWO_ADDRESSES)

    void    I2C_SetAddress2(uint8 address ) ;
    uint8   I2C_GetAddress2(void) ;
    void    I2C_SetBuffer2(uint16 bufSize, uint16 rwBoundry, volatile uint8 * dataPtr) ;

#endif  /* (I2C_ADDRESSES == I2C_TWO_ADDRESSES) */


/* EZI2C interrupt handler */
CY_ISR_PROTO(I2C_ISR);
#if(I2C_WAKEUP_ENABLED)
    CY_ISR_PROTO(I2C_WAKEUP_ISR);
#endif /* (I2C_WAKEUP_ENABLED) */

/***************************************
*     Vars with External Linkage
***************************************/

extern uint8 I2C_initVar;


/***************************************
*   Initial Parameter Constants
***************************************/

#define I2C_DEFAULT_ADDR1      (8u)
#define I2C_DEFAULT_ADDR2      (9u)

#define I2C_BUS_SPEED          (100u)
#define I2C_SUBADDR_WIDTH      (0u)
#define I2C_SUBADDR_8BIT       (0x00u) /* 8-bit sub-address width */
#define I2C_SUBADDR_16BIT      (0x01u) /* 16-bit sub-address width */
#define I2C_BUS_PORT           (1u)


/***************************************
*              API Constants
***************************************/

/* Status bit definition */

/* A read addr 1 operation occured since last status check */
#define I2C_STATUS_READ1       (0x01u)

/* A Write addr 1 opereation occured since last status check */
#define I2C_STATUS_WRITE1      (0x02u)

/* A read addr 2 operation occured since last status check */
#define I2C_STATUS_READ2       (0x04u)

/* A Write addr 2 opereation occured since last status check */
#define I2C_STATUS_WRITE2      (0x08u)

/* A start has occured, but a Stop has not been detected */
#define I2C_STATUS_BUSY        (0x10u)

/* Addr 1 read busy */
#define I2C_STATUS_RD1BUSY     (0x11u)

/* Addr 1 write busy */
#define I2C_STATUS_WR1BUSY     (0x12u)

/* Addr 2 read busy */
#define I2C_STATUS_RD2BUSY     (0x14u)

/* Addr 2 write busy */
#define I2C_STATUS_WR2BUSY     (0x18u)

/* Mask for status bits. */
#define I2C_STATUS_MASK        (0x1Fu)

/* An Error occured since last read */
#define I2C_STATUS_ERR         (0x80u)

/* Dummy data to be sent to master */
#define I2C_DUMMY_DATA         (0xFFu)

/* The I2C Master bits in I2C cinfiguration register */
#define I2C_I2C_MASTER_MASK    (0xDDu)

/* Component's enable/disable state */
#define I2C_ENABLED            (0x01u)
#define I2C_DISABLED           (0x00u)

#define I2C_BUS_SPEED_50KHZ      (50u)

/* Bus speed grater 50kHz requires 16 oversample rate */
#if (I2C_BUS_SPEED <= I2C_BUS_SPEED_50KHZ)

	#define I2C_OVER_SAMPLE_RATE       (32u)

#else

	#define I2C_OVER_SAMPLE_RATE       (16u)

#endif  /* End (I2C_BUS_SPEED <= I2C_BUS_SPEED_50KHZ) */

/* Divide factor calculation */
#define I2C_DIVIDE_FACTOR_WITH_FRACT_BYTE  \
                (((uint32) BCLK__BUS_CLK__KHZ << 8u) / ((uint32)I2C_BUS_SPEED * \
                I2C_OVER_SAMPLE_RATE))
                
#define I2C_DIVIDE_FACTOR  (((uint32) I2C_DIVIDE_FACTOR_WITH_FRACT_BYTE) >> 8u)


/* Returns 1 if corresponding bit is set, otherwise 0 */
#define I2C_IS_BIT_SET(value, mask) (((mask) == ((value) & (mask))) ? 0x01u : 0x00u)

#define I2C_ADDRESS_SHIFT      (1u)
#define I2C_ADDRESS_LSB_SHIFT  (8u)


/***************************************
*              Registers
***************************************/

/* I2C Extended Configuration Register */
#define I2C_XCFG_REG       (* (reg8 *) I2C_I2C_Prim__XCFG )
#define I2C_XCFG_PTR       (  (reg8 *) I2C_I2C_Prim__XCFG )

/* I2C Slave Adddress Register */
#define I2C_ADDR_REG       (* (reg8 *) I2C_I2C_Prim__ADR )
#define I2C_ADDR_PTR       (  (reg8 *) I2C_I2C_Prim__ADR )

/* I2C Configuration Register */
#define I2C_CFG_REG        (* (reg8 *) I2C_I2C_Prim__CFG )
#define I2C_CFG_PTR        (  (reg8 *) I2C_I2C_Prim__CFG )

/* I2C Control and Status Register */
#define I2C_CSR_REG        (* (reg8 *) I2C_I2C_Prim__CSR )
#define I2C_CSR_PTR        (  (reg8 *) I2C_I2C_Prim__CSR )

/* I2C Data Register */
#define I2C_DATA_REG       (* (reg8 *) I2C_I2C_Prim__D )
#define I2C_DATA_PTR       (  (reg8 *) I2C_I2C_Prim__D )

/*  8 LSB bits of the 10-bit Clock Divider */
#define I2C_CLKDIV1_REG        (* (reg8 *) I2C_I2C_Prim__CLK_DIV1 )
#define I2C_CLKDIV1_PTR        (  (reg8 *) I2C_I2C_Prim__CLK_DIV1 )

/* 2 MSB bits of the 10-bit Clock Divider */
#define I2C_CLKDIV2_REG        (* (reg8 *) I2C_I2C_Prim__CLK_DIV2 )
#define I2C_CLKDIV2_PTR        (  (reg8 *) I2C_I2C_Prim__CLK_DIV2 )

/* Power System Control Register 1 */
#define I2C_PWRSYS_CR1_REG     (* (reg8 *) CYREG_PWRSYS_CR1 )
#define I2C_PWRSYS_CR1_PTR     (  (reg8 *) CYREG_PWRSYS_CR1 )

/* I2C operation in Active Mode */
#define I2C_PM_ACT_CFG_REG     (* (reg8 *) I2C_I2C_Prim__PM_ACT_CFG )
#define I2C_PM_ACT_CFG_PTR     (  (reg8 *) I2C_I2C_Prim__PM_ACT_CFG )

/* I2C operation in Alternate Active (Standby) Mode */
#define I2C_PM_STBY_CFG_REG    (* (reg8 *) I2C_I2C_Prim__PM_STBY_CFG )
#define I2C_PM_STBY_CFG_PTR    (  (reg8 *) I2C_I2C_Prim__PM_STBY_CFG )


/***************************************
*       Register Constants
***************************************/

/* XCFG I2C Extended Configuration Register */
#define I2C_XCFG_CLK_EN            (0x80u)
#define I2C_XCFG_HDWR_ADDR_EN      (0x01u)

/* Force nack */
#define I2C_XCFG_FORCE_NACK        (0x10u)

/* Ready to sleep */
#define I2C_XCFG_SLEEP_READY       (0x20u)

/* Should be set before entering sleep mode */
#define I2C_XCFG_I2C_ON            (0x40u)

/* Enables the I2C regulator backup */
#define I2C_PWRSYS_CR1_I2C_BACKUP  (0x04u)

/* Data I2C Slave Data Register */
#define I2C_SADDR_MASK        (0x7Fu)
#define I2C_DATA_MASK         (0xFFu)
#define I2C_READ_FLAG         (0x01u)

/* CFG I2C Configuration Register */

/* Pin Select for SCL/SDA lines */
#define I2C_CFG_SIO_SELECT    (0x80u)

/* Pin Select */
#define I2C_CFG_PSELECT       (0x40u)

/* Bus Error Interrupt Enable */
#define I2C_CFG_BUS_ERR_IE    (0x20u)

/* Enable Interrupt on STOP condition */
#define I2C_CFG_STOP_IE       (0x10u)

/* Enable Interrupt on STOP condition */
#define I2C_CFG_STOP_ERR_IE   (0x10u)

/* Clock rate mask. 1 for 50K, 0 for 100K and 400K */
#define I2C_CFG_CLK_RATE      (0x04u)


/* Enable Slave operation */
#define I2C_CFG_EN_SLAVE      (0x01u)

/* CSR I2C Control and Status Register */

/* Active high when bus error has occured */
#define I2C_CSR_BUS_ERROR     (0x80u)

/* Set to 1 if lost arbitration in host mode */
#define I2C_CSR_LOST_ARB      (0x40u)

/* Set to 1 if Stop has been detected */
#define I2C_CSR_STOP_STATUS   (0x20u)

/* ACK response */
#define I2C_CSR_ACK           (0x10u)

/* NAK response */
#define I2C_CSR_NAK           (0x00u)

/* Set in firmware 0 = status bit, 1 Address is slave */
#define I2C_CSR_ADDRESS       (0x08u)

/* Set in firmware 1 = transmit, 0 = receive. */
#define I2C_CSR_TRANSMIT      (0x04u)

/* Last received bit. */
#define I2C_CSR_LRB           (0x02u)

 /* Last received bit was an ACK */
#define I2C_CSR_LRB_ACK       (0x00u)

/* Last received bit was an NAK */
#define I2C_CSR_LRB_NAK       (0x02u)

/* Informs that last byte has been sent. */
#define I2C_CSR_BYTE_COMPLETE (0x01u)

/* CLK_DIV I2C Clock Divide Factor Register */

/* Status bit, Set at Start and cleared at Stop condition */
#define I2C_CLK_DIV_MSK       (0x07u)

/* Divide input clock by  1 */
#define I2C_CLK_DIV_1         (0x00u)

/* Divide input clock by  2 */
#define I2C_CLK_DIV_2         (0x01u)

/* Divide input clock by  4 */
#define I2C_CLK_DIV_4         (0x02u)

/* Divide input clock by  8 */
#define I2C_CLK_DIV_8         (0x03u)

/* Divide input clock by 16 */
#define I2C_CLK_DIV_16        (0x04u)

/* Divide input clock by 32 */
#define I2C_CLK_DIV_32        (0x05u)

/* Divide input clock by 64 */
#define I2C_CLK_DIV_64        (0x06u)

/* Active Power Mode CFG Register - power enable mask */
#define I2C_ACT_PWR_EN    I2C_I2C_Prim__PM_ACT_MSK

/* Alternate Active (Standby) Power Mode CFG Register - power enable mask */
#define I2C_STBY_PWR_EN    I2C_I2C_Prim__PM_STBY_MSK

/* Number of the I2C_isr interrupt. */
#define I2C_ISR_NUMBER    I2C_isr__INTC_NUMBER

/* Priority of the I2C_isr interrupt. */
#define I2C_ISR_PRIORITY  I2C_isr__INTC_PRIOR_NUM

/* I2C state machine constants */

/* Wait for Start */
#define  I2C_SM_IDLE              (0x00u)

/* Default address states */

/* Wait for sub-address */
#define  I2C_SM_DEV1_WR_ADDR      (0x01u)

/* Wait for sub-address MSB */
#define  I2C_SM_DEV1_WR_ADDR_MSB  (0x01u)

/* Wait for sub-address LSB */
#define  I2C_SM_DEV1_WR_ADDR_LSB  (0x02u)

/* Get data from Master */
#define  I2C_SM_DEV1_WR_DATA      (0x04u)

/* Send data to Master */
#define  I2C_SM_DEV1_RD_DATA      (0x08u)

/* Second address states */

/* Wait for sub-address */
#define  I2C_SM_DEV2_WR_ADDR      (0x11u)

/* Wait for sub-address MSB */
#define  I2C_SM_DEV2_WR_ADDR_MSB  (0x11u)

/* Wait for sub-address LSB */
#define  I2C_SM_DEV2_WR_ADDR_LSB  (0x12u)

/* Get data from Master */
#define  I2C_SM_DEV2_WR_DATA      (0x14u)

/* Send data to Master */
#define  I2C_SM_DEV2_RD_DATA      (0x18u)


/***************************************
*       Obsolete
***************************************/

/* Following definitions are OBSOLETE and must not be used */

#if(I2C_ADDRESSES == I2C_ONE_ADDRESS)

    void    I2C_SlaveSetSleepMode(void) ;
    void    I2C_SlaveSetWakeMode(void) ;

#endif /* (I2C_ADDRESSES == I2C_ONE_ADDRESS)*/

#define I2C_State          I2C_curState
#define I2C_Status         I2C_curStatus
#define I2C_DataPtr        I2C_dataPtrS1

#define I2C_RwOffset1      I2C_rwOffsetS1
#define I2C_RwIndex1       I2C_rwIndexS1
#define I2C_WrProtect1     I2C_wrProtectS1
#define I2C_BufSize1       I2C_bufSizeS1

#if(I2C_ADDRESSES == I2C_TWO_ADDRESSES)

    #define I2C_DataPtr2   I2C_dataPtrS2
    #define I2C_Address1   I2C_addrS1
    #define I2C_Address2   I2C_addrS2

    #define I2C_RwOffset2  I2C_rwOffsetS2
    #define I2C_RwIndex2   I2C_rwIndexS2
    #define I2C_WrProtect2 I2C_wrProtectS2
    #define I2C_BufSize2   I2C_bufSizeS2

#endif /* I2C_ADDRESSES == I2C_TWO_ADDRESSES */


#endif /* CY_EZI2C_I2C_H */


/* [] END OF FILE */
