/* =============================================================================
 *
 * Capsense_Monitor.c
 *
 * This task periodically scans the CapSense widgets, reads the results into a
 * CS_PACKET variable and (if the data are meaningful) posts them to a queue.
 *
 * v1.0	YFS	Initial release. 
 *
 * =============================================================================
*/

/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>
#include <Task_Defs.h>


/* Declare the queue for the CapSense events */
QueueHandle_t CapSense_Monitor_Q;

/* Declare a local array of packets to be posted to the queue */
static CS_PACKET packet[CAPSENSE_MONITOR_Q_SIZE];
static int packetindex = 0;

/*
 * Function:	CapSense_Monitor_Task
 *
 * This function runs as a FreeRTOS task with a high priority. It scans the
 * CapSense buttons and sliders and posts the results to a message queue.
 * In order to reduce the number of context switches needed between this
 * task and those consuming the data only new button presses are posted.
 * Scans yielding no presses or no changes are not posted.
 *
 *			Start task
 *			Start CapSense
 *		 -> Update baselines and scan widgets
 *		|	Delay 1ms while CapSense is busy (as needed, but probably just once)
 *		|	Read the scan results into static circular buffer
 *		|   If results are "interesting" (new and non-zero)...
 *		|   	Post result to the queue
 *		|   	Move to next buffer entry
 *		|   	Delay to allow client tasks to get and use the data in the queue
 *		 -- Loop
 * 
 * Globals:		Uses CapSense_Monitor_Q
 * 				Writes to local packet array and counter
 *
 * Parameters:	pvParameters (not used)
 *
 * Return:		None (infinite loop)
 */

void CapSense_Monitor_Task( void *pvParameters )
{
    BaseType_t msgSent;             /* Return from queue API */
    
	uint16 sliderval;		        /* Hold the slider state and value */
		
	CS_PACKET newpacket = { 0 };	/* Hold the packet returned from the queue */
	
	uint32 lastpacket = 0;	        /* Used to prevent re-sends of prior events */

	/* Prepare the CapSense component for scanning */
	CapSense_Start();
    CapSense_InitializeAllBaselines();
		
	for( ; ; )		/* Main scanning loop (forever) */
	{	
        /* Update baselines for all the sensors and start a scan */
        CapSense_UpdateEnabledBaselines();
        CapSense_ScanEnabledWidgets();

		while( CapSense_IsBusy() )
		{
			/*
			Wait for CapSense to complete scanning. This takes about 500us for
			2 buttons and a 5-element slider. Minimum OS delay is 1ms.
			*/
			vTaskDelay( 1 );
		}

		/* Scanning is complete - read the widgets and populate the packet */
		newpacket.b0 = CapSense_CheckIsWidgetActive( CapSense_BUTTON0__BTN );
        newpacket.b1 = CapSense_CheckIsWidgetActive( CapSense_BUTTON1__BTN );
		
        sliderval = CapSense_GetCentroidPos( CapSense_LINEARSLIDER0__LS );
		
		if( sliderval == NO_FINGER )
		{
			newpacket.slider = 0;       /* Slider not touched */
			newpacket.sliderval = 0;
		}
		else
		{
			newpacket.slider = 1;       /* Slider touched */
			newpacket.sliderval = sliderval;		
		}

		/*
		Only post the result if something is pressed.
		newpacket.result != 0			- says a button was pressed
		newpacket.result != lastpacket	- says the state changed (de-bounce)
		*/
		if( newpacket.result != 0 && newpacket.result != lastpacket  )
		{
			/* Copy the interesting packet into the data for the queue */
			packet[packetindex].result = newpacket.result;
			
			/* Make sure the queue never gets full */
            if( uxQueueMessagesWaiting( CapSense_Monitor_Q ) == CAPSENSE_MONITOR_Q_SIZE )
            {
                xQueueReset( CapSense_Monitor_Q );
            }
            		
			/* Post the packet via the queue */
            msgSent = xQueueSend(
                CapSense_Monitor_Q,
                (void *)&packet[packetindex].result,
                DO_NOT_WAIT_ON_QUEUE );
            
            if( msgSent != pdTRUE )
            {
                report_error( "Queue send" );
            }
			
			/* Move the pointer to the next array element (circular) */
			packetindex++;
			packetindex %= CAPSENSE_MONITOR_Q_SIZE;
			
		} /* if( newpacket.result != 0 && newpacket.result != lastpacket  ) */
		
		/* Remember last packet so one press is not reported multiple times */
		lastpacket = newpacket.result;
		
		/* Go to sleep to let the client tasks extract the data and run */
        vTaskDelay( 20 );   /* 20ms delay - ~50 CapSense reads/sec is plenty */
	
	} /* for( ; ; ) */

} /* CapSense_Monitor_Task() */
