/* =============================================================================
 *
 * MainTask.c
 *
 * This file contains the code for the main (menu) task (Main_Task) and the application tasks.
 *
 * (c) 2015 by Marilyn Wolf
 *
 * =============================================================================
*/

/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>
#include <Task_Defs.h>

/* Declare the event groups for Main_Task and the demos */
EventGroupHandle_t Demo_Events;
EventGroupHandle_t Main_Task_Event;

/* Declare an array of demo names. Used to display the demo menu */
static char * const demoNames[NUM_DEMOS] =
{
	READER_TASK,
	WRITER_TASK,
};

/*
 * Function:		Main_Task
 *
 * This function runs as a FreeRTOS task with a low priority. It creates the
 * Reader and Writer tasks, event groups, and the CapSense monitor task and queue. The main
 * loop displays the menu of demos and allows them to be browsed via CapSense
 * buttons (up/down) and started (SW2).
 *

 * 
 * Globals:		Uses Demo_Events, Main_Task_Event and CapSense_Monitor_Q
 *
 * Parameters:	pvParameters (not used)
 *
 * Return:		None (infinite loop)
 */
void Main_Task( void *pvParameters )
{
	CS_PACKET demo;		    /* Hold a message from CapSense Q */
       
    BaseType_t queueDataReceived;   /* Return from xQueueCreate */
    
    BaseType_t taskCreated;         /* Return from xTaskCreate */
	
	int demoIndex = 0;	    /* Currently selected demo (0 means demo #1) */

	int initScreen = 1;	    /* Flag to show the intro text on LCD */
	
	/* Start the LCD and print the welcome text */ 
	LCD_Start();
	print_welcome();
    	
	/*
	Create the OS resources used by the applications
	*/
	
	/* Create the CapSense_queue - must create it prior to tasks that use it */
    CapSense_Monitor_Q = xQueueCreate(
        CAPSENSE_MONITOR_Q_SIZE,
        sizeof( CS_PACKET ) );
    
    if( ! CapSense_Monitor_Q )
    {
        report_error( "Queue create" );
    }
    
    /* Create the event group to control which demo to run */
    Demo_Events = xEventGroupCreate();
    
    if( ! Demo_Events )
    {
        report_error( "Event create" );
    }
    
    /* Create the event group that stops this task from running while the
    demos are started but inactive */
    Main_Task_Event = xEventGroupCreate();
    
    if( ! Main_Task_Event )
    {
        report_error( "Event create" );
    }
    
    /*
    Create the application tasks and CapSense monitor
    */
    
	/* Create Reader_Task */   
	taskCreated = xTaskCreate(
        Reader_Task,
        READER_TASK,
        DEMO_TASK_STACK_SIZE,
        0,          /* Pass the index as a task argument*/
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );
    
    if( taskCreated != pdPASS )
    {
        report_error( "Reader task" );
    }    
	
	/* Create Writer_Task */
	taskCreated = xTaskCreate(
        Writer_Task,
        WRITER_TASK,
        DEMO_TASK_STACK_SIZE,
        0,          
        DEMO_TASK_PRIORITY,
        TASK_HANDLE_UNUSED );
    
    if( taskCreated != pdPASS )
    {
        report_error( "Writer task" );
    } 
 
	/* Reset demoIndex to the start of the array */
    demoIndex = 0;
    
	/* Flush any early scan data from the CapSense queue */
    xQueueReset( CapSense_Monitor_Q );
	
	/* By this point, the OS is running and all tasks have been created */
	
	/* Print out the instructions for using the demo */		
	scroll( 1, "Reader/Writer Demo" );
	
	for( ; ; )		/* Main loop (forever) */
	{
		
		/* Collect the CapSense activity from the message queue */
        queueDataReceived = xQueueReceive(
            CapSense_Monitor_Q,
            &demo.result,           /* Put queue data here */
            DO_NOT_WAIT_ON_QUEUE );
        
        if( queueDataReceived == pdTRUE )        
		{
			/* Move the index based on the button press */
			if( demo.b0 )
				demoIndex--;		/* Move up */
			else if( demo.b1 )
				demoIndex++;		/* Move down */
			
			/* Make the menu circular */
			if( demoIndex < MIN_DEMO )
				demoIndex = MAX_DEMO;
			else if( demoIndex > MAX_DEMO )
				demoIndex = MIN_DEMO;	
				
			/* Clear the LCD and display the new demo name */
			show_demo_title( demoIndex, demoNames[demoIndex] );
		} /* if( queueDataReceived == pdTRUE ) */
		
	} /* for( ; ; ) */

} /* Main_Task() */
